/**
 * 
 *
 * Software: LGExtract
 * Authors: Matthieu Constant and Elsa Tolone
 *
 * Copyright (C) 2010-2011 Université Paris-Est Marne-la-Vallée
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.
 *
 */

package fr.umlv.lgextract.lobjects;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.jdom.Element;

import fr.umlv.lgextract.LGExtractFormatting;
import fr.umlv.lgextract.exceptions.LGExtractException;
import fr.umlv.lgextract.exceptions.LGExtractNoEntryException;
import fr.umlv.lgextract.table.LGTableEntry;

/**
 * 
 * @author Matthieu Constant
 *
 */

public class StringFeature extends AbstractLingObj {
	private static char inTableOfTablesPropertyPrefix = '*';
	private final String attribute;
	private final Set<String> values = new HashSet<String>();
	
	public StringFeature(String attribute){
		this.attribute = attribute;
		setType(attribute);
	}
	
	public StringFeature(String attribute,String values){
		this.attribute = attribute;
		setValue(values);
		setType(attribute);
	}
	
	private void setValue(String value) {
		String[] values = value.split("\\|");
		this.values.clear();
		for(String v : values){
	      this.values.add(v);
		}
	}
	
	public String getAttribute() {
		return attribute;
	}

	private boolean isProperty(String s){
		return (s.startsWith("@") && s.endsWith("@"));
	}
	
	private boolean isInTableOfTablesOnly(String prop){
		return prop.charAt(1) == inTableOfTablesPropertyPrefix;
	}
	
	private String getPropertyName(String s){
		if(isInTableOfTablesOnly(s)){
			return s.substring(2,s.length() - 1);
		}
		return s.substring(1,s.length() - 1);
	}
	
	private String normalize(String s){
//		 Compile regular expression
        Pattern pattern = Pattern.compile(" +");        
        // Replace all occurrences of pattern in input
        Matcher matcher = pattern.matcher(s);
        s = matcher.replaceAll(" ");
        pattern = Pattern.compile("(^ +)|( +$)");
        matcher = pattern.matcher(s);
        s = matcher.replaceAll("");
		return s;
	}
	
	@Override
	public void resolve(LGTableEntry entry, LGExtractLingObjects objects) throws LGExtractException,LGExtractNoEntryException {
		HashSet<String> set = new HashSet<String>();
		HashSet<String> toRemove = new HashSet<String>();
		Pattern p = Pattern.compile("(@[^@]+@)|([^@]+)");		 
		
		for(String originVal : values){
			String v = originVal.substring(1,originVal.length()-1);
			Matcher m = p.matcher(v);	
			String res = "";
			while(m.find()){
				String va = m.group();
				if(isProperty(va)){					
					String val = entry.getValue(getPropertyName(va),isInTableOfTablesOnly(va));
					if(!val.equals("false")){
						res += val;
					}
				}
				else{
					res += va;
				}				
			}
			if(!res.equals(v)){
				toRemove.add(originVal);
				if(!res.equals("")){
					res = normalize(res);
					//res.replaceAll(" +", " ");
					set.add("\""+res+"\"");
				}
			}
		}
		values.removeAll(toRemove);
		values.addAll(set);		
	}

	@Override
	public void addIn(FeatureStructure fs) throws LGExtractException {
		StringFeature f = (StringFeature)fs.get(this.getType());
		if(f != null){
			this.addValues( f.getValues());
		}
		fs.getMap().put(this.getType(), this);
	}
	
	
	public void addValues(Collection<String> values){
		this.values.addAll(values);
	}
	
	
	@Override
	public List<LingObj> getObjects(String id) throws LGExtractException{
		ArrayList<LingObj> list = new ArrayList<LingObj>();
		if(!getAttribute().equals(id)){
			return list;
		}		
		list.add(this);
		return list;
	}	
	
	
	public Set<String> getValues() {
		return values;
	}
	
	
	@Override
	public LingObj clone(LGExtractLingObjects objects) {
		String name = getName();
		StringFeature f;
		if((name == null) || ((f = (StringFeature)objects.getByName(name)) == null)){
			f = new StringFeature(getAttribute());
			f.setName(getName());
			f.setType(getType());
		}
		f.addValues(getValues());
		return f;
	}
	
	@Override
	public LingObj emptyInstance() {
		StringFeature f = new StringFeature(getAttribute());
		f.setName(getName());
		f.setType(getType());
		return f;
	}
	
	
	public String toString(){
		String s = "";
		s += this.getAttribute()+"=";
		Iterator<String> it = values.iterator(); 
		while(it.hasNext()){
			s += it.next().toString();
			if(it.hasNext()){
				s += "|";
			}
		}
		return s;
	}	
    
	@Override
	public void toXML(Element parent, LGExtractFormatting formatting) {
		String val = "";
		Iterator<String> it = values.iterator(); 
		while(it.hasNext()){
			String s = it.next().toString(); 
			val += s.substring(1,s.length() - 1);
			if(it.hasNext()){
				val += "|";
			}
		}
		
		if(formatting.isElementAttribute(parent.getName(), getAttribute())){			
			parent.setAttribute(getAttribute(),val);
			return;
		}
		Element par = new Element(getAttribute());
		parent.addContent(par);		
		par.setAttribute("value",val);		
	}
	
}
